<?php
/*
  This action sorts all Wacko associative lists. It only sorts lists where all items
  are assoc, other lists are left as is.

  Parameters:
    * collation - a space-separated list of locale names to try and use when comparing
      strings (for setlocale/strcoll); if neither of them can be set {{SortAssocList}}
      outputs a warning in place where it's called;
      # NOTE: since UWiki uses UTF-8 and locale strings typically look like "ru_RU.UTF-8"
              trailing ".UTF-8" part is added utomatically so you can simply pass "ru_RU".
      * if you're on *nix you can get the list of locations by doing "locale -a".
    * by - values: text (default), raw. This first compares textual values while the
      second compares source markup. For example, for two items: "**bold**" and
      "apriore" - by=text takes "bold" and "apriore" where the latter goes the first
      while by=raw takes "*..." and "a..." and the former comes before the latter;
      * this action uses 'on before render' hook when by=text and 'on after parsing'
        when by=raw so if you want it to sort render-time-created assoc lists too pass
        it "on render" flag.
    * on render - makes this action run on render time (see by=raw examplation above);
    * type - values: str (default), int, float;
    * desc - sorts items in descending order (desc=0 sorts in ascending, by default).

  This action affects all lists in current document so you can't sort some of them using
  one set of settings and others using a different setup. However, you can create nested
  documents with %%(wacko) formatter:

    {{SortAssocLists type=int, desc}}
    = 123 == This list is sorted as integers in descending order.
    %%(wacko)
      {{SortAssocLists by=raw}}
      = **and this list** is sorted by default (as strings) based on items' raw markup.
    %%

  CONFIGURATION - fields added to UWikiSettings:

    * $AssocListSortingCollation - default value used when "collation" argument is omitted.
      If undefined default PHP locale is used.
*/

class Usortassoclists_Root extends UWikiBaseAction {
  public $params, $isDescSorting;

  function IsDynamic($format, $params) {
    return !empty($params['on render']) or empty($params['by']) or $params['by'] !== 'raw';
  }

  function Priority($format, $params) { return (int) UWikiMaxPriority * 0.2; }

  function Execute($format, $params) {
    $lists = &$format->root->doc->elements['wacko_SubList'];
    if ($lists) {
      $this->params = $params + array('by' => 'value',  'type' => 'str');
      $this->isDescSorting = !empty($params['desc']);

      $oldLocale = setlocale(LC_COLLATE, '0');

        $coll = &$params['collation'];
        $coll or $coll = &$this->settings->AssocListSortingCollation;
        if (!$this->SetLocale($coll)) {
          $format->appendMe = true;
          $error = $this->children[] = $this->NewElement('Usortassoclists_WrongLocale');
          $error->format = array($params['collation'], $oldLocale);
        }

      try {
        foreach ($lists as $list) {
          $allAssoc = true;

          foreach ($list->children as $item) {
            if ($item->markerType !== 'assoc') {
              $allAssoc = false;
              break;
            }
          }

          $allAssoc and usort($list->children, array($this, 'Sorter'));
        }

        setlocale(LC_COLLATE, $oldLocale);
      } catch (Exception $e) {
        setlocale($oldLocale);
        throw $e;
      }
    }
  }

    function SetLocale($locStr) {
      if (trim($locStr)) {
        $locales = explode(' ', $locStr);
        foreach ($locales as &$loc) {
          $loc = trim($loc);
          if (strtolower(substr($loc, -6)) !== '.utf-8') {
            $loc .= '.UTF-8';
          }
        }

        array_unshift($locales, LC_COLLATE);
        return call_user_func_array('setlocale', $locales);
      } else {
        return true;
      }
    }

    function Sorter($item1, $item2) {
      $value1 = $this->ValueOf($item1, $numSuffix1);
      $value2 = $this->ValueOf($item2, $numSuffix2);

      if ($numSuffix1 === false) {
        $result = strcoll($value1, $value2);
      } else {
        $result = ($value1 > $value2 ? +1 : ($value1 < $value2 ? -1 : strcoll($numSuffix1, $numSuffix2)));
      }

      $this->isDescSorting and $result *= -1;
      return $result;
    }

      function ValueOf(Uwacko_ListItem $obj, &$suffix) {
        switch ($this->params['by']) {
        case 'raw':     $value = $obj->originalRaw; break;
        default:        $value = strip_tags($obj->TermElement()->ChildrenToHTML()); break;
        }

        switch ($this->params['type']) {
        case 'int':     $value = self::ExtractNumFrom($value, $suffix, false); break;
        case 'float':   $value = self::ExtractNumFrom($value, $suffix, true); break;
        default:        $suffix = false;
        }

        return $value;
      }

  // taken from iTools' `sort` text script command.
  static function ExtractNumFrom(&$str, &$suffix, $float = false) {
    // PCRE is probably faster for UTF-8 strings than manually traversing it wtith mb_substr() and co.
    $float and $float = ',.';
    if (preg_match('/([+-]?[0-9'.$float.']+)(.*)/uS', $str, $match)) {
      $suffix = $match[2];

      $num = str_replace(',', '.', $match[1]);
      return $float ? ((float) $num) : ((int) $num);
    } else {
      EUWikiLastPCRE::ThrowIfPcreFailed();
      $suffix = $str;
      return 0;
    }
  }
}

  class Usortassoclists_WrongLocale extends UWikiFormatErrorMessage {
    public $message = '{{sortassoclists: wrong locale';
    public $defaultMessage = '{{SortAssocLists}} could not set locale to %s - using default (%s).';
  }
